/*******************************************************************
 * Fritz Fun                                                       *
 * Created by Jan-Michael Brummer                                  *
 * All parts are distributed under the terms of GPLv2. See COPYING *
 *******************************************************************/

/**
 * \file voxplay.c
 * \brief Play voice box speex files from FRITZ!Box
 */

#include <ffgtk.h>
#include <speex/speex_callbacks.h>

#define MAX_FRAME_SIZE	2000

static FILE *psFile = NULL;
static short anOutput[ MAX_FRAME_SIZE ];
static spx_int32_t nFrameSize = 0;
static void *psSpeex = NULL;
static SpeexBits nBits;
static spx_int32_t nRate = 0;
static unsigned char nBytes = 0;
static char anPacketBuffer[ 38 ];
static int nJ, nNumRead;
static const SpeexMode *pnMode;
static struct sAudio *psAudio = NULL;
static void *pPriv = NULL;
static gboolean bStop = FALSE;
static GtkWidget *psDialog = NULL;
static GtkWidget *psLabel;
static GtkWidget *psContentArea;

/**
 * \brief Main playback thread
 * \param pData UNUSED
 * \return NULL
 */
static gpointer playbackThread( gpointer pData ) {
	bStop = FALSE;

	while ( !feof( psFile ) && bStop == FALSE ) {
		nNumRead = fread( &nBytes, sizeof( char ), 1, psFile );

		if ( nBytes != 0x26 ) {
			continue;
		}

		nNumRead = fread( anPacketBuffer, sizeof( char ), ( int ) nBytes, psFile );
		if ( nNumRead != sizeof( char ) * nBytes ) {
			break;
		}
		speex_bits_read_from( &nBits, ( char * ) anPacketBuffer, nBytes );

		for ( nJ = 0; nJ != 2; nJ++ ) {
			int nRet;

			nRet = speex_decode_int( psSpeex, &nBits, anOutput );
			if ( nRet == -1 ) {
				break;
			} else if ( nRet == -2 ) {
				fprintf( stderr, "Decoding error: corrupted stream?\n" );
				break;
			}

			if ( speex_bits_remaining( &nBits ) < 0 ) {
				fprintf( stderr, "Decoding overflow: corrupted stream?\n" );
				break;
			}
		}

		psAudio -> Write( pPriv, ( unsigned char * ) anOutput, nFrameSize * sizeof( short ) );
	}

	speex_decoder_destroy( psSpeex );
	speex_bits_destroy( &nBits );

	fclose( psFile );

	return NULL;
}

/**
 * \brief Delete event
 * \param psWidget button widget
 * \param psEvent gdk event pointer
 * \param pData UNUSED
 */
static gboolean deleteEvent( GtkWidget *psWidget, GdkEvent *psEvent, gpointer pData ) {
	Debug( KERN_DEBUG, "called\n" );

	Debug( KERN_DEBUG, "close call\n" );
	bStop = TRUE;
	psAudio -> Close( pPriv, TRUE );

	//gtk_widget_destroy( psDialog );
	psDialog = NULL;

	return FALSE;
}

/**
 * \brief Play voicebox message file
 * \param pnFile message file
 * \return error code
 */
gint PlayVoxFile( gchar *pnFile ) {
	if ( psDialog != NULL ) {
		return -1;
	}

	psAudio = getDefaultAudioPlugin( getActiveProfile() );
	if ( psAudio == NULL ) {
		return -1;
	}

	pPriv = psAudio -> Open();
	if ( pPriv == NULL ) {
		Debug( KERN_DEBUG, "Could not open audio device\n" );
		return -2;
	}

	/* open file read-only binary */
	psFile = fopen( pnFile, "rb" );
	if ( psFile == NULL ) {
		psAudio -> Close( pPriv, FALSE );
		return -3;
	}

	speex_bits_init( &nBits );

	nRate = 8000;

	pnMode = speex_lib_get_mode( 0 );

	psSpeex = speex_decoder_init( pnMode );
	if ( psSpeex == NULL ) {
		fprintf( stderr, "Decoder initialization failed.\n" );
		psAudio -> Close( pPriv, FALSE );
		return -1;
	}
	speex_decoder_ctl( psSpeex, SPEEX_GET_FRAME_SIZE, &nFrameSize );
	speex_decoder_ctl( psSpeex, SPEEX_SET_SAMPLING_RATE, &nRate );

	/* Create dialog */
	psDialog = gtk_dialog_new_with_buttons( _( "Voice box" ), NULL, GTK_DIALOG_MODAL, GTK_STOCK_CLOSE, GTK_RESPONSE_CLOSE, NULL );
	psContentArea = gtk_dialog_get_content_area( GTK_DIALOG( psDialog ) );
	psLabel = gtk_label_new( _( "Playing voice box entry..." ) );
	gtk_container_add( GTK_CONTAINER( psContentArea ), psLabel );
	g_signal_connect_swapped( psDialog, "response", G_CALLBACK( gtk_widget_destroy ), psDialog );
	g_signal_connect( psDialog, "destroy", G_CALLBACK( deleteEvent ), psDialog );
	gtk_widget_set_size_request( psDialog, 300, 150 );
	gtk_window_set_position( GTK_WINDOW( psDialog ), GTK_WIN_POS_CENTER );

	gtk_widget_show_all( GTK_WIDGET( psDialog ) );

	CREATE_THREAD( "playback", playbackThread, NULL );

	return 0;
}
