// Copyright 2014 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "chrome/browser/chromeos/file_system_provider/operations/read_file.h"

#include <stddef.h>

#include <limits>
#include <string>
#include <utility>

#include "base/trace_event/trace_event.h"
#include "chrome/common/extensions/api/file_system_provider.h"
#include "chrome/common/extensions/api/file_system_provider_internal.h"

namespace chromeos {
namespace file_system_provider {
namespace operations {
namespace {

// Convert |value| into |output|. If parsing fails, then returns a negative
// value. Otherwise returns number of bytes written to the buffer.
int CopyRequestValueToBuffer(scoped_ptr<RequestValue> value,
                             scoped_refptr<net::IOBuffer> buffer,
                             int buffer_offset,
                             int buffer_length) {
  using extensions::api::file_system_provider_internal::
      ReadFileRequestedSuccess::Params;

  const Params* params = value->read_file_success_params();
  if (!params)
    return -1;

  const size_t chunk_size = params->data.size();

  // Check for overflows.
  if (chunk_size > static_cast<size_t>(buffer_length) - buffer_offset)
    return -1;

  memcpy(buffer->data() + buffer_offset, params->data.data(), chunk_size);

  return chunk_size;
}

}  // namespace

ReadFile::ReadFile(
    extensions::EventRouter* event_router,
    const ProvidedFileSystemInfo& file_system_info,
    int file_handle,
    scoped_refptr<net::IOBuffer> buffer,
    int64_t offset,
    int length,
    const ProvidedFileSystemInterface::ReadChunkReceivedCallback& callback)
    : Operation(event_router, file_system_info),
      file_handle_(file_handle),
      buffer_(buffer),
      offset_(offset),
      length_(length),
      current_offset_(0),
      callback_(callback) {}

ReadFile::~ReadFile() {
}

bool ReadFile::Execute(int request_id) {
  using extensions::api::file_system_provider::ReadFileRequestedOptions;
  TRACE_EVENT0("file_system_provider", "ReadFile::Execute");

  ReadFileRequestedOptions options;
  options.file_system_id = file_system_info_.file_system_id();
  options.request_id = request_id;
  options.open_request_id = file_handle_;
  options.offset = offset_;
  options.length = length_;

  return SendEvent(
      request_id,
      extensions::events::FILE_SYSTEM_PROVIDER_ON_READ_FILE_REQUESTED,
      extensions::api::file_system_provider::OnReadFileRequested::kEventName,
      extensions::api::file_system_provider::OnReadFileRequested::Create(
          options));
}

void ReadFile::OnSuccess(int /* request_id */,
                         scoped_ptr<RequestValue> result,
                         bool has_more) {
  TRACE_EVENT0("file_system_provider", "ReadFile::OnSuccess");
  const int copy_result = CopyRequestValueToBuffer(std::move(result), buffer_,
                                                   current_offset_, length_);

  if (copy_result < 0) {
    LOG(ERROR) << "Failed to parse a response for the read file operation.";
    callback_.Run(
        0 /* chunk_length */, false /* has_more */, base::File::FILE_ERROR_IO);
    return;
  }

  if (copy_result > 0)
    current_offset_ += copy_result;
  callback_.Run(copy_result, has_more, base::File::FILE_OK);
}

void ReadFile::OnError(int /* request_id */,
                       scoped_ptr<RequestValue> /* result */,
                       base::File::Error error) {
  TRACE_EVENT0("file_system_provider", "ReadFile::OnError");
  callback_.Run(0 /* chunk_length */, false /* has_more */, error);
}

}  // namespace operations
}  // namespace file_system_provider
}  // namespace chromeos
