from __future__ import with_statement
import unittest
import os
from os import path as op
import send2trash.plat_other
from send2trash.plat_other import send2trash as s2t
from ConfigParser import ConfigParser
from tempfile import mkdtemp, NamedTemporaryFile, mktemp
import shutil
import stat
from io import open
# Could still use cleaning up. But no longer relies on ramfs.

def touch(path):
  with open(path, u'a'):
    os.utime(path, None)

class TestHomeTrash(unittest.TestCase):
  def setUp(self):
    self.file = NamedTemporaryFile(dir=op.expanduser(u"~"), 
      prefix=u'send2trash_test', delete=False)

  def test_trash(self):
    s2t(self.file.name)
    self.assertFalse(op.exists(self.file.name))

  def tearDown(self):
    hometrash = send2trash.plat_other.HOMETRASH
    name = op.basename(self.file.name)
    os.remove(op.join(hometrash, u'files', name))
    os.remove(op.join(hometrash, u'info', name+u'.trashinfo'))

#
# Tests for files on some other volume than the user's home directory.
#
# What we need to stub:
# * plat_other.get_dev (to make sure the file will not be on the home dir dev)
# * os.path.ismount (to make our topdir look like a top dir)
#
class TestExtVol(unittest.TestCase):
  def setUp(self):
    self.trashTopdir = mkdtemp(prefix=u's2t')
    self.fileName = u'test.txt'
    self.filePath = op.join(self.trashTopdir, self.fileName)
    touch(self.filePath)

    self.old_ismount = old_ismount = op.ismount
    self.old_getdev = send2trash.plat_other.get_dev
    def s_getdev(path):
      from send2trash.plat_other import is_parent
      st = os.lstat(path)
      if is_parent(self.trashTopdir, path):
        return u'dev'
      return st
    def s_ismount(path):
      if op.realpath(path) == op.realpath(self.trashTopdir):
        return True
      return old_ismount(path)

    send2trash.plat_other.os.path.ismount = s_ismount
    send2trash.plat_other.get_dev = s_getdev

  def tearDown(self):
    send2trash.plat_other.get_dev = self.old_getdev
    send2trash.plat_other.os.path.ismount = self.old_ismount
    shutil.rmtree(self.trashTopdir)

class TestTopdirTrash(TestExtVol):
  def setUp(self):
    TestExtVol.setUp(self)
    # Create a .Trash dir w/ a sticky bit
    self.trashDir = op.join(self.trashTopdir, u'.Trash')
    os.mkdir(self.trashDir, 0777|stat.S_ISVTX)

  def test_trash(self):
    s2t(self.filePath)
    self.assertFalse(op.exists(self.filePath))
    self.assertTrue(op.exists(op.join(self.trashDir, unicode(os.getuid()), u'files', self.fileName)))
    self.assertTrue(op.exists(op.join(self.trashDir, unicode(os.getuid()), u'info', self.fileName + u'.trashinfo')))
    # info relative path (if another test is added, with the same fileName/Path,
    # then it gets renamed etc.)
    cfg = ConfigParser()
    cfg.read(op.join(self.trashDir, unicode(os.getuid()), u'info', self.fileName + u'.trashinfo'))
    self.assertEqual(self.fileName, cfg.get(u'Trash Info', u'Path', 1))

# Test .Trash-UID
class TestTopdirTrashFallback(TestExtVol):
  def test_trash(self):
    touch(self.filePath)
    s2t(self.filePath)
    self.assertFalse(op.exists(self.filePath))
    self.assertTrue(op.exists(op.join(self.trashTopdir, u'.Trash-' + unicode(os.getuid()), u'files', self.fileName)))

# Test failure
class TestTopdirFailure(TestExtVol):
  def setUp(self):
    TestExtVol.setUp(self)
    os.chmod(self.trashTopdir, 0500) # not writable to induce the exception

  def test_trash(self):
    with self.assertRaises(OSError):
      s2t(self.filePath)
    self.assertTrue(op.exists(self.filePath))

  def tearDown(self):
    os.chmod(self.trashTopdir, 0700) # writable to allow deletion
    TestExtVol.tearDown(self)

# Make sure it will find the mount point properly for a file in a symlinked path
class TestSymlink(TestExtVol):
  def setUp(self):
    TestExtVol.setUp(self)
    # Use mktemp (race conditioney but no symlink equivalent)
    # Since is_parent uses realpath(), and our getdev uses is_parent,
    # this should work
    self.slDir = mktemp(prefix=u's2t', dir=op.expanduser(u'~'))
    
    os.mkdir(op.join(self.trashTopdir, u'subdir'), 0700)
    self.filePath = op.join(self.trashTopdir, u'subdir', self.fileName)
    touch(self.filePath)
    os.symlink(op.join(self.trashTopdir, u'subdir'), self.slDir)

  def test_trash(self):
    s2t(op.join(self.slDir, self.fileName))
    self.assertFalse(op.exists(self.filePath))
    self.assertTrue(op.exists(op.join(self.trashTopdir, u'.Trash-' + unicode(os.getuid()), u'files', self.fileName)))

  def tearDown(self):
    os.remove(self.slDir)
    TestExtVol.tearDown(self)

if __name__ == u'__main__':
  unittest.main()
