//  ************************************************************************************************
//
//  BornAgain: simulate and fit reflection and scattering
//
//! @file      GUI/View/Main/MainWindow.h
//! @brief     Defines class MainWindow
//!
//! @homepage  http://www.bornagainproject.org
//! @license   GNU General Public License v3 or higher (see COPYING)
//! @copyright Forschungszentrum Jülich GmbH 2018
//! @authors   Scientific Computing Group at MLZ (see CITATION, AUTHORS)
//
//  ************************************************************************************************

#ifndef BORNAGAIN_GUI_VIEW_MAIN_MAINWINDOW_H
#define BORNAGAIN_GUI_VIEW_MAIN_MAINWINDOW_H

#include "GUI/Support/Data/ID.h"
#include <QMainWindow>

class ActionManager;
class ImportDataView;
class InstrumentView;
class JobView;
class ProjectManager;
class ProjectsView;
class SampleView;
class SimulationView;

class QButtonGroup;
class QProgressBar;
class QStackedLayout;
class QStatusBar;
class QToolButton;
class QVBoxLayout;

class MainWindow : public QMainWindow {
    Q_OBJECT

public:
    explicit MainWindow();
    ~MainWindow() override;

    QProgressBar* progressBar();

    ProjectManager* projectManager();

    QWidget* currentView() const;
    void setCurrentView(int viewId);

    void updateTitle();
    void loadProject(QString projectPath);

public slots:
    void onFocusRequest(int index);
    void openRecentProject();
    void onRunSimulationShortcut();

signals:
    void currentViewChanged(GUI::ID::ViewId newView);

protected:
    void closeEvent(QCloseEvent* event) override;

private:
    void initApplication();
    void initProgressBar();
    void initButtons();
    void initViews();

    void readSettings();
    void writeSettings();
    void addButton(GUI::ID::ViewId id, const QIcon& icon, const QString& title,
                   const QString& tooltip);
    void resetView(GUI::ID::ViewId id, QWidget* view);
    QToolButton* createViewSelectionButton() const;

    //! Recalculate the size of the view selection buttons to show complete button text
    void updateViewSelectionButtonsGeometry() const;
    void onDocumentOpenedOrClosed(bool open);
    void onDocumentModified();
    void onAboutToCloseDocument();

    void raiseView(int viewId);

    QProgressBar* m_progressBar;
    QButtonGroup* m_viewSelectionButtons;
    QStackedLayout* m_viewsStack;
    QVBoxLayout* m_viewSelectionButtonsLayout;

    ProjectManager* m_projectManager;
    ActionManager* m_actionManager;

    ProjectsView* m_projectsView;
    InstrumentView* m_instrumentView;
    SampleView* m_sampleView;
    ImportDataView* m_importDataView;
    SimulationView* m_simulationView;
    JobView* m_jobView;
};

#endif // BORNAGAIN_GUI_VIEW_MAIN_MAINWINDOW_H
